﻿using System;
// ReSharper disable RedundantUsingDirective
using System.ComponentModel;
// ReSharper restore RedundantUsingDirective
using System.Diagnostics;
using System.IO;
using System.Windows.Forms;
using WMPLib;

namespace AudioTranscription
{
	public partial class mainTranscriber : Form
	{
		WMPLib.WindowsMediaPlayer _player;

		private int jumpAmount = 5;

		private void PlayFile(String url)
		{
			if (_player == null)
				_player = new WMPLib.WindowsMediaPlayer();

			_player.settings.autoStart = false;

			_player.StatusChange += _player_StatusChange;
			_player.PositionChange += _player_PositionChange;
			_player.PlayStateChange += _player_PlayStateChange;
			_player.CurrentMediaItemAvailable += _player_CurrentMediaItemAvailable;

			_player.URL = url;
		}

		void _player_CurrentMediaItemAvailable(string bstrItemName)
		{
			Debug.WriteLine(_player.currentMedia.duration);
			togglePlayPause.Enabled = true;
		}

		void _player_PlayStateChange(int NewState)
		{
			WMPLib.WMPPlayState playState = (WMPLib.WMPPlayState) NewState;

			if (playState == WMPPlayState.wmppsReady)
			{
				Debug.WriteLine(_player.currentMedia.duration);
				togglePlayPause.Enabled = true;
				this.statusLabel.Text = Path.GetFileName(_player.currentMedia.sourceURL) + " ready to play...";
				this.Text = "Transcriber: " + Path.GetFileName(_player.currentMedia.sourceURL);

			}

			if (_player.currentMedia != null && _player.currentMedia.duration > 0)
			{
				//TimeSpan videoLength = TimeSpan.FromSeconds(_player.currentMedia.duration);
				videoProgress.Enabled = true;
	
			}


			if (playState == WMPPlayState.wmppsPlaying)
			{
				this.togglePlayPause.Text = "Pause";
			}
			else
			{
				this.togglePlayPause.Text = "Play";
			}
		}

		void _player_StatusChange()
		{
			Debug.WriteLine("Status Change");
		}

		void _player_PositionChange(double oldPosition, double newPosition)
		{
			Debug.WriteLine(_player.currentMedia.duration);
		}

		public mainTranscriber()
		{
			InitializeComponent();
			this.KeyPreview = true;
	
		}

		private void btnLoadRecording_Click(object sender, EventArgs e)
		{
			string fileName;

			if (openAudioFile.ShowDialog() == DialogResult.OK)
			{
				fileName = openAudioFile.FileName;

				PlayFile(fileName);
				this.statusLabel.Text = Path.GetFileName(fileName) + " loaded...";
			}
		}

		private void copyText_Click(object sender, EventArgs e)
		{
			CopyToClipboard();
			this.statusLabel.Text = "transcript copied to the clipboard...";
		}

		private void CopyToClipboard()
		{
			transcript.SelectAll();
			transcript.Copy();
		}

		private void togglePlayPause_Click(object sender, EventArgs e)
		{
			WMPLib.WMPPlayState playState = _player.playState;


			if (playState == WMPPlayState.wmppsPlaying)
			{
				_player.controls.pause();
			}
			else
			{
				_player.controls.play();
			}

		}

		protected override void OnKeyDown(System.Windows.Forms.KeyEventArgs args)
		{
			if ((args.KeyCode == Keys.Tab))
			{
				args.Handled = true;
				args.SuppressKeyPress = true;
			}
			else if (args.KeyCode == Keys.OemPipe)
			{
				args.Handled = true;
				args.SuppressKeyPress = true;
				HandleHotKey(false);
			}

			base.OnKeyDown(args);
		}

		private void HandleHotKey(bool togglePause)
		{
			if (_player != null)
			{
				WMPLib.WMPPlayState playState = _player.playState;


				if (playState == WMPPlayState.wmppsPlaying && togglePause)
				{
					//if audio is currently playing, pause;
					_player.controls.pause();
				}
				else
				{
					//if audio is not playing, then jump back jumpAmount seconds and play;
					if (_player.controls.currentPosition > jumpAmount)
						_player.controls.currentPosition -= jumpAmount;
					else
						_player.controls.currentPosition = 0;

					if (togglePause)
						_player.controls.play();
				}
			}

		}

		protected override bool ProcessTabKey(bool forward)
		{
			HandleHotKey(true);
			return false;
		}

		private void mainTranscriber_KeyDown(object sender, KeyEventArgs e)
		{
		}

		private void positionTimer_Tick(object sender, EventArgs e)
		{
			if (_player !=  null && _player.currentMedia != null && _player.currentMedia.duration > 0)
			{
				//TimeSpan currentPosition = TimeSpan.FromSeconds(_player.controls.currentPosition);
				currentPosition.Text = _player.controls.currentPositionString;
				double percentage = Math.Truncate((_player.controls.currentPosition/_player.currentMedia.duration)*100);
				videoProgress.Value = Convert.ToInt32(percentage);
			}

		}

		private void videoProgress_Scroll(object sender, EventArgs e)
		{

			double desiredPosition = (_player.currentMedia.duration * videoProgress.Value) / 100.0;
			_player.controls.currentPosition = desiredPosition;

		}

		private void videoProgress_ValueChanged(object sender, EventArgs e)
		{
			
			//double desiredPosition = (_player.currentMedia.duration*videoProgress.Value)/100.0;
			//_player.controls.currentPosition = desiredPosition;

		}

		private string lastSavedText = "";

		private void autoSave_Tick(object sender, EventArgs e)
		{
			this.statusLabel.Text = "autosaving...";
			AutoSaveTranscript();
			this.statusLabel.Text = "";
		}

		private void AutoSaveTranscript()
		{
			if (transcript.TextLength > 0 && transcript.Text != lastSavedText)
			{
				bool saveSuccessful = false;
				string fileName = GetAutoSaveFileName();
				int saveTry = 0;

				while (!saveSuccessful)
				{
					try
					{
						transcript.SaveFile(fileName, RichTextBoxStreamType.RichText);
						saveSuccessful = true;
					}
					catch (IOException)
					{
						fileName = Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.MyDocuments),
						                        String.Format("transcribe_autosave_{0}.rtf", saveTry));
						saveTry++;
					}
				}

				lastSavedText = transcript.Text;
			}
		}

		private static string GetAutoSaveFileName()
		{
			return Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.MyDocuments),
			                    "transcribe_autosave.rtf");
		}

		private void mainTranscriber_FormClosing(object sender, FormClosingEventArgs e)
		{
			if (transcript.TextLength > 0)
			{
				AutoSaveTranscript();

				DialogResult result =
					MessageBox.Show(
						"You have a transcript in progress, would you like to copy it to the clipboard before closing this application?",
						"Copy Transcript?", MessageBoxButtons.YesNoCancel, MessageBoxIcon.Question, MessageBoxDefaultButton.Button3);

				if (result == DialogResult.Yes)
				{
					CopyToClipboard();
				}
				else if (result == DialogResult.Cancel)
				{
					e.Cancel = true;
				}
			}
		}

		private void mainTranscriber_Load(object sender, EventArgs e)
		{
			//load in auto-jump amount (default to 5)

			jumpAmount = Properties.Settings.Default.SecondsToJump;

			secondsToJump.Value = jumpAmount;
			

			string fileName = GetAutoSaveFileName();

			if (File.Exists(fileName))
			{
				DialogResult shouldLoad = MessageBox.Show("An 'in-progress' transcript file was found, do you wish to load it?",
				                                          "Auto Save Found", MessageBoxButtons.YesNo, MessageBoxIcon.Question,
				                                          MessageBoxDefaultButton.Button1);

				if (shouldLoad == DialogResult.Yes)
				{
					while (shouldLoad == DialogResult.Yes)
					{						
						try
						{
							transcript.LoadFile(fileName);
							shouldLoad = DialogResult.No;
						}
						catch (Exception ex)
						{
							DialogResult retry = MessageBox.Show("Error loading auto-save: " + ex.Message, "Error loading Auto Save", MessageBoxButtons.RetryCancel,
											MessageBoxIcon.Error, MessageBoxDefaultButton.Button1);

							if (retry == DialogResult.Cancel)
								shouldLoad = DialogResult.No;
						}
					}
				}
			}
		}

		private void saveTranscript_Click(object sender, EventArgs e)
		{
			if (transcript.TextLength == 0)
			{
				MessageBox.Show("Nothing to save", "Save Canceled", MessageBoxButtons.OK, MessageBoxIcon.Information);
			}
			else
			{
				saveTranscriptFile.InitialDirectory = Environment.GetFolderPath(Environment.SpecialFolder.MyDocuments);
				if (saveTranscriptFile.ShowDialog() == DialogResult.OK)
				{
					string fileName = saveTranscriptFile.FileName;

					transcript.SaveFile(fileName, RichTextBoxStreamType.RichText);
				}
			}
		}

		private void secondsToJump_ValueChanged(object sender, EventArgs e)
		{
			if (secondsToJump.Value != jumpAmount)
			{
				jumpAmount = Convert.ToInt32(secondsToJump.Value);
				Properties.Settings.Default.SecondsToJump = jumpAmount;
				Properties.Settings.Default.Save();
			}
		}
	}
}
